<?php

use Illuminate\Auth\Events\Lockout;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Livewire\Attributes\Layout;
use Livewire\Attributes\Validate;
use Livewire\Volt\Component;

new #[Layout('components.layouts.auth')] class extends Component {
    #[Validate('required|string|email')]
    public string $email = '';

    #[Validate('required|string')]
    public string $password = '';

    public bool $remember = false;

    /**
     * Handle an incoming authentication request.
     */
    public function login(): void
    {
        $this->validate();

        $this->ensureIsNotRateLimited();

        if (! Auth::attempt(['email' => $this->email, 'password' => $this->password], $this->remember)) {
            RateLimiter::hit($this->throttleKey());

            throw ValidationException::withMessages([
                'email' => __('auth.failed'),
            ]);
        }

        RateLimiter::clear($this->throttleKey());
        Session::regenerate();

        $this->redirectIntended(default: route('dashboard', absolute: false), navigate: true);
    }

    /**
     * Ensure the authentication request is not rate limited.
     */
    protected function ensureIsNotRateLimited(): void
    {
        if (! RateLimiter::tooManyAttempts($this->throttleKey(), 5)) {
            return;
        }

        event(new Lockout(request()));

        $seconds = RateLimiter::availableIn($this->throttleKey());

        throw ValidationException::withMessages([
            'email' => __('auth.throttle', [
                'seconds' => $seconds,
                'minutes' => ceil($seconds / 60),
            ]),
        ]);
    }

    /**
     * Get the authentication rate limiting throttle key.
     */
    protected function throttleKey(): string
    {
        return Str::transliterate(Str::lower($this->email).'|'.request()->ip());
    }
}; ?>
<div class="min-h-screen flex  justify-center bg-gradient-to-br from-blue-100 via-white to-indigo-100">
    <div class="w-full max-w-md bg-white rounded-2xl shadow-xl p-8 space-y-8 animate-fade-in-down">
        
        <!-- Logo animado -->
        <div class="flex justify-center">
            <img 
                src="{{ asset('img/kardiafit.png') }}" 
                alt="Logo" 
                class="h-24 animate-bounce drop-shadow-md"
            >
        </div>

        <!-- Encabezado -->
        <x-auth-header 
            title="Bienvenido de nuevo" 
            description="Ingresa tu RUT y contraseña para continuar"
        />

        <!-- Estado de la sesión -->
        <x-auth-session-status class="text-center" :status="session('status')" />

        <!-- Formulario -->
        <form wire:submit="login" class="flex flex-col gap-5">
            <!-- RUT -->
            <flux:input
                wire:model="rut"
                label="RUT"
                type="text"
                required
                autofocus
                autocomplete="username"
                placeholder="12.345.678-9"
            />

            <!-- Contraseña -->
            <div class="relative">
                <flux:input
                    wire:model="password"
                    label="Contraseña"
                    type="password"
                    required
                    autocomplete="current-password"
                    placeholder="Tu contraseña secreta"
                    viewable
                />

                @if (Route::has('password.request'))
                    <flux:link 
                        class="absolute right-0 top-0 text-xs text-blue-600 hover:underline" 
                        :href="route('password.request')" 
                        wire:navigate
                    >
                        ¿Olvidaste tu contraseña?
                    </flux:link>
                @endif
            </div>

            <!-- Recuérdame -->
            <flux:checkbox wire:model="remember" label="Recordarme la próxima vez" />

            <!-- Botón -->
            <div class="pt-2">
                <flux:button variant="primary" type="submit" class="w-full text-lg">
                    🚀 Iniciar sesión
                </flux:button>
            </div>
        </form>

        <!-- Enlace de registro -->
        @if (Route::has('register'))
            <div class="text-center text-sm text-zinc-600 dark:text-zinc-400">
                ¿Aún no tienes cuenta?
                <flux:link 
                    :href="route('register')" 
                    wire:navigate 
                    class="text-blue-600 hover:underline font-semibold"
                >
                    ¡Regístrate gratis!
                </flux:link>
            </div>
        @endif
    </div>
</div>

